<?php
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');
include("csism_db.php");
date_default_timezone_set('Asia/Kolkata');
header('Content-Type: application/json');


// Path to your Firebase service account JSON
$serviceAccountFile = __DIR__ . '/service-account.json';
$serviceAccount = json_decode(file_get_contents($serviceAccountFile), true);

// Read JSON input
$data = json_decode(file_get_contents('php://input'), true);

if (empty($data['attendance'])) {
    echo json_encode(['success' => false, 'message' => 'No attendance data provided.']);
    exit;
}

// Step 2: Generate OAuth 2.0 access token manually
$jwt = createJWT($serviceAccount);
$accessToken = getAccessToken($jwt);

if (!$accessToken) {
    echo json_encode(['success' => false, 'error' => 'Failed to get access token']);
    exit;
}

$notifications = [];

register_shutdown_function(function() {
    $error = error_get_last();
    if ($error !== null) {
        http_response_code(200); // force 200 so Flutter can decode
        echo json_encode(['success' => false, 'message' => $error['message']]);
    }
});

/**
 * Helper: compute attendance status (keeps your original logic but works with full datetime strings)
 * Returns: [$statusString, $statusTypeId]
 */
function getAttendanceStatus($shiftIn, $shiftOut, $graceMin, $checkIn = null, $checkOut = null) {
    $status = "Present";
    $statusTypeId = 1;

    $halfDayTime = date('Y-m-d H:i:s', strtotime("+4 hours", strtotime($shiftIn)));
    $allowedCheckIn = date('Y-m-d H:i:s', strtotime("+$graceMin minutes", strtotime($shiftIn)));

    $isLate = false;
    $isFirstHalfAbsent = false;
    $isSecondHalfEarly = false;
    $isFirstHalfPresentSecondHalfAbsent = false;
    $isSecondHalfAbsentDueToNoCheckout = false;

    // --- Check IN ---
    if ($checkIn) {
        if (strtotime($checkIn) > strtotime($allowedCheckIn) && strtotime($checkIn) <= strtotime($halfDayTime)) {
            $isLate = true; // Late
        }
        if (strtotime($checkIn) > strtotime($halfDayTime)) {
            $isFirstHalfAbsent = true; // First half absent
        }
    }

    // --- Check OUT ---
    if ($checkOut) {
        if (strtotime($checkOut) < strtotime($shiftOut) && strtotime($checkOut) >= strtotime($halfDayTime)) {
            $isSecondHalfEarly = true; // Left early but after half day
        }
        if (strtotime($checkOut) < strtotime($halfDayTime)) {
            $isFirstHalfPresentSecondHalfAbsent = true; // Out before half-day cutoff
        }
    } else {
        // If checkout is null AND current time already passed shift out
        if (time() > strtotime($shiftOut) && $checkIn) {
            $isSecondHalfAbsentDueToNoCheckout = true;
        }
    }

    // --- Final Status ---
    if (!$checkIn && !$checkOut) {
        $status = "Absent"; $statusTypeId = 2;
    }
    elseif ($isFirstHalfAbsent && $isSecondHalfEarly) {
        $status = "1st HalfAbsent + 2nd HalfEarly Going"; $statusTypeId = 1;
    }
    elseif ($isLate && $isSecondHalfEarly) {
        $status = "Late + 2nd HalfEarly Going"; $statusTypeId = 6;
    }
    elseif ($isLate) {
        $status = "Late"; $statusTypeId = 6;
    }
    elseif ($isSecondHalfEarly) {
        $status = "2nd HalfEarly Going"; $statusTypeId = 1;
    }
    elseif ($isFirstHalfAbsent) {
        $status = "1st HalfAbsent"; $statusTypeId = 2;
    }
    elseif ($isFirstHalfPresentSecondHalfAbsent) {
        $status = "1st HalfPresent, 2nd HalfAbsent"; $statusTypeId = 1;
    }
    elseif ($isSecondHalfAbsentDueToNoCheckout) {
        $status = "2nd HalfAbsent"; $statusTypeId = 1;
    }
    else {
        $status = "Present"; $statusTypeId = 1;
    }

    return [$status, $statusTypeId];
}

foreach ($data['attendance'] as $record) {
    $user_id   = isset($record['user_code']) ? intval($record['user_code']) : '';
    $latitude  = isset($record['latitude']) ? floatval($record['latitude']) : 0.0;
    $longitude = isset($record['longitude']) ? floatval($record['longitude']) : 0.0;
    $currentDate = date('Y-m-d');
    $currentTime = date('Y-m-d H:i:s');
    $statusType = strtoupper($record['type'] ?? 'IN');
    
    $shiftQuery = $conn->prepare("SELECT a.shift_in, a.shift_out, a.grace_min, a.shift_mode, b.name, b.token FROM shift_master a INNER JOIN staff_master b ON a.shift_id=b.shift_id WHERE b.staff_id = ? ORDER BY a.shift_id DESC LIMIT 1");
    
    $shiftQuery->bind_param("i", $user_id);
    $shiftQuery->execute();
    $shiftResult = $shiftQuery->get_result();
    $shift = $shiftResult->fetch_assoc();
    $shiftQuery->close();

    if (!$shift) {
        echo json_encode(['success' => false, 'message' => 'Shift not found for user.']);
        exit;
    }

    // Build DateTime objects accounting for night shifts (shift_out possibly next day)
    $shiftMode = strtolower($shift['shift_mode']); // 'day' or 'night'
    // shift times are like "09:00:00" etc in the DB. Combine with today's date to form datetimes.
    $shiftInDT = DateTime::createFromFormat('Y-m-d H:i:s', $currentDate . ' ' . $shift['shift_in']);
    $shiftOutDT = DateTime::createFromFormat('Y-m-d H:i:s', $currentDate . ' ' . $shift['shift_out']);
    $username = $shift['name'];
    $token = $shift['token'] ?? '';

    if ($shiftMode === 'night') {
        // If shift_out time is earlier or equal than shift_in time => it belongs to the next day
        if (strtotime($shift['shift_out']) <= strtotime($shift['shift_in'])) {
            $shiftOutDT->modify('+1 day');
        }
    } else {
        // day shift: if timezone or data causes shift_out to be before shift_in (unlikely), normalize by adding a day
        if ($shiftOutDT <= $shiftInDT) {
            $shiftOutDT->modify('+1 day');
        }
    }

    $shiftIn = $shiftInDT->format('Y-m-d H:i:s');
    $shiftOut = $shiftOutDT->format('Y-m-d H:i:s');
    $graceMin  = (int)$shift['grace_min'];

    // shift_date should represent the date of shift start
    $shiftStartDate = $shiftInDT->format('Y-m-d');

    // get last record
    $lastQuery = $conn->prepare("SELECT id, check_in_time, check_out_time, shift_date 
                                 FROM attendance 
                                 WHERE user_id = ? ORDER BY id DESC LIMIT 1");
    $lastQuery->bind_param("i", $user_id);
    $lastQuery->execute();
    $lastResult = $lastQuery->get_result();
    $lastRow = $lastResult->fetch_assoc();
    $lastQuery->close();

    // --- DAY SHIFT ---
    if ($shiftMode === 'day') {
        if ($statusType === 'IN') {
            // If current time is after shift_out, disallow check-in (shift already finished)
            if (strtotime($currentTime) > strtotime($shiftOut)) {
                echo json_encode(['success' => false, 'message' => 'Cannot check IN after shift end.']);
                exit;
            }

            list($status, $statusId) = getAttendanceStatus($shiftIn, $shiftOut, $graceMin, $currentTime, null);

            // Allow new IN only if lastRow is absent OR different date OR last has check_out_time filled
            if (!$lastRow || $lastRow['shift_date'] != $shiftStartDate || !empty($lastRow['check_out_time'])) {
                $stmt = $conn->prepare("INSERT INTO attendance 
                    (user_id, latitude, longitude, shift_date, time, check_in_time, status_type_id, attendance_status) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
                    
                // types: i d d s s s i s
                $stmt->bind_param("iddsssis", $user_id, $latitude, $longitude, $shiftStartDate, $currentTime, $currentTime, $statusId, $status);
                $stmt->execute();
                $stmt->close();
                
                $title = "Check-IN";
                    $body = "Hi {$username}, your check IN time is {$currentTime}";
                    
                    $payload = [
                        "message" => [
                            "token" => $token,
                            "notification" => [
                                "title" => $title,
                                "body" => $body
                            ],
                            "data" => [
                               "click_action" => "FLUTTER_NOTIFICATION_CLICK",
                               "with_actions" => "false"
                            ]
                        ]
                    ];
                    
                    $payloadJson = json_encode($payload, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
                    $Isread = 'false';
                    $type = 'Attendance';
                    // --- 1. Insert into database first ---
                    $stmt = $conn->prepare("INSERT INTO Notification (staff_id, noti_type, noti_title, noti_message, Is_read, create_datetime) VALUES (?, ?, ?, ?, ?, ?)");
                    $stmt->bind_param("ssssss", $user_id, $type,  $title, $body, $Isread, $currentTime);
                    $stmt->execute();
                    $stmt->close();
                    
                    // --- 2. Send the notification ---
                    $sent = sendFCMv1($payloadJson, $accessToken, $serviceAccount['project_id']);
                    
                    // --- 3. Optionally log send status ---
                    $notifications[] = [
                        "staff" => $username,
                        "status" => $sent
                    ];
            } else {
                echo json_encode(['success' => false, 'message' => 'Already checked IN today.']);
                exit;
            }

        } elseif ($statusType === 'OUT') {
            if ($lastRow && empty($lastRow['check_out_time'])) {
                $lastShiftDate = $lastRow['shift_date'];
        
                // Allow checkout if same shift start date OR next day of last shift date
                if ($lastShiftDate == $shiftStartDate || 
                    date('Y-m-d', strtotime($lastShiftDate . ' +1 day')) == $shiftStartDate) {
                    
                    list($status, $statusId) = getAttendanceStatus(
                        $shiftIn, $shiftOut, $graceMin, 
                        $lastRow['check_in_time'], $currentTime
                    );
        
                    $updateStmt = $conn->prepare("UPDATE attendance 
                        SET check_out_time = ?, attendance_status = ?, status_type_id = ? WHERE id = ?");
                    $updateStmt->bind_param("ssii", $currentTime, $status, $statusId, $lastRow['id']);
                    $updateStmt->execute();
                    $updateStmt->close();
                    
                    $title = "Check-OUT";
                    $body = "Hi {$username}, your check OUT time is {$currentTime}";
                    
                    $payload = [
                        "message" => [
                            "token" => $token,
                            "notification" => [
                                "title" => $title,
                                "body" => $body
                            ],
                            "data" => [
                               "click_action" => "FLUTTER_NOTIFICATION_CLICK",
                               "with_actions" => "false"
                            ]
                        ]
                    ];
                    
                    $payloadJson = json_encode($payload, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
                   
                    $Isread = 'false';
                    $type = 'Attendance';
                    // --- 1. Insert into database first ---
                    $stmt = $conn->prepare("INSERT INTO Notification (staff_id, noti_type, noti_title, noti_message, Is_read, create_datetime) VALUES (?, ?, ?, ?, ?, ?)");
                    $stmt->bind_param("ssssss", $user_id, $type,  $title, $body, $Isread, $currentTime);
                    $stmt->execute();
                    $stmt->close();
                    
                    // --- 2. Send the notification ---
                    $sent = sendFCMv1($payloadJson, $accessToken, $serviceAccount['project_id']);
                    
                    // --- 3. Optionally log send status ---
                    $notifications[] = [
                        "staff" => $username,
                        "status" => $sent
                    ];

                } else {
                    echo json_encode(['success' => false, 'message' => 'No active IN record to checkout (night shift).']);
                    exit;
                }
            } else {
                echo json_encode(['success' => false, 'message' => 'No active IN record to checkout (night shift).']);
                exit;
            }
        }
    }

    // --- NIGHT SHIFT ---
    elseif ($shiftMode === 'night') {
        if ($statusType === 'IN') {
            // For night shift check: if IN occurs after overall shiftOut (i.e., too late), disallow.
            if (strtotime($currentTime) > strtotime($shiftOut)) {
                echo json_encode(['success' => false, 'message' => 'Cannot check IN after shift end (night shift).']);
                exit;
            }

            list($status, $statusId) = getAttendanceStatus($shiftIn, $shiftOut, $graceMin, $currentTime, null);

            // Allow new IN only if lastRow missing OR lastRow has checkout filled
            if (!$lastRow || !empty($lastRow['check_out_time']) || $lastRow['shift_date'] != $shiftStartDate) {
                $stmt = $conn->prepare("INSERT INTO attendance 
                    (user_id, latitude, longitude, shift_date, time, check_in_time, status_type_id, attendance_status) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
                // types: i d d s s s i s
                $stmt->bind_param("iddsssis", $user_id, $latitude, $longitude, $shiftStartDate, $currentTime, $currentTime, $statusId, $status);
                $stmt->execute();
                $stmt->close();
            } else {
                echo json_encode(['success' => false, 'message' => 'Already checked IN (night shift).']);
                exit;
            }

        } elseif ($statusType === 'OUT') {
            // Validate lastRow exists, belongs to this shiftStartDate and has empty checkout
            if ($lastRow && empty($lastRow['check_out_time']) && $lastRow['shift_date'] == $shiftStartDate) {
                list($status, $statusId) = getAttendanceStatus($shiftIn, $shiftOut, $graceMin, $lastRow['check_in_time'], $currentTime);

                $updateStmt = $conn->prepare("UPDATE attendance 
                    SET check_out_time = ?, attendance_status = ?, status_type_id = ? WHERE id = ?");
                $updateStmt->bind_param("ssii", $currentTime, $status, $statusId, $lastRow['id']);
                $updateStmt->execute();
                $updateStmt->close();
            } else {
                echo json_encode(['success' => false, 'message' => 'No active IN record to checkout (night shift).']);
                exit;
            }
        }
    }
}


// ----------------- FUNCTIONS -----------------

function base64UrlEncode($data) {
    return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
}

function createJWT($serviceAccount) {
    $header = ['alg' => 'RS256', 'typ' => 'JWT'];
    $now = time();
    $claimSet = [
        'iss' => $serviceAccount['client_email'],
        'scope' => 'https://www.googleapis.com/auth/firebase.messaging',
        'aud' => $serviceAccount['token_uri'],
        'iat' => $now,
        'exp' => $now + 3600
    ];

    $base64UrlHeader = base64UrlEncode(json_encode($header));
    $base64UrlClaimSet = base64UrlEncode(json_encode($claimSet));

    $signatureInput = $base64UrlHeader . "." . $base64UrlClaimSet;

    openssl_sign($signatureInput, $signature, $serviceAccount['private_key'], "SHA256");

    $base64UrlSignature = base64UrlEncode($signature);

    return $signatureInput . "." . $base64UrlSignature;
}

function getAccessToken($jwt) {
    $tokenUri = 'https://oauth2.googleapis.com/token';
    $postFields = http_build_query([
        'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
        'assertion' => $jwt
    ]);

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $tokenUri);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/x-www-form-urlencoded'
    ]);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $postFields);

    $response = curl_exec($ch);
    curl_close($ch);

    $data = json_decode($response, true);
    return $data['access_token'] ?? null;
}

function sendFCMv1($payloadJson, $accessToken, $projectId) {
    $url = "https://fcm.googleapis.com/v1/projects/{$projectId}/messages:send";

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        "Authorization: Bearer {$accessToken}",
        "Content-Type: application/json"
    ]);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $payloadJson); // send JSON string directly

    $response = curl_exec($ch);
    if ($response === FALSE) {
        $error = curl_error($ch);
        curl_close($ch);
        return $error;
    }
    curl_close($ch);
    return $response;
}

$conn->close();
echo json_encode(['success' => true, 'message' => 'Attendance synced successfully.']);
?>
