<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

header('Access-Control-Allow-Origin: *');
header('Content-Type: application/json');

include("csism_db.php");
date_default_timezone_set('Asia/Kolkata');

// Path to your Firebase service account JSON
$serviceAccountFile = __DIR__ . '/service-account.json';
$serviceAccount = json_decode(file_get_contents($serviceAccountFile), true);

// Step 1: Get staff who did NOT check in today in attendance table
$query = "
    SELECT 
        a.staff_id, 
        a.name, 
        b.shift_in, 
        a.token 
    FROM staff_master a
    INNER JOIN shift_master b ON a.shift_id = b.shift_id
    LEFT JOIN attendance c 
        ON a.staff_id = c.user_id AND DATE(c.check_in_time) = CURDATE()
    WHERE b.shift_in < NOW() 
      AND a.token IS NOT NULL
";

$result = mysqli_query($conn, $query);
if (!$result) {
    echo json_encode(['success' => false, 'error' => mysqli_error($conn)]);
    exit;
}

// Step 2: Generate OAuth 2.0 access token manually
$jwt = createJWT($serviceAccount);
$accessToken = getAccessToken($jwt);

if (!$accessToken) {
    echo json_encode(['success' => false, 'error' => 'Failed to get access token']);
    exit;
}

$notifications = [];

while ($row = mysqli_fetch_assoc($result)) {
    $staffId = $row['staff_id'] ?? '';
    $staffName = $row['name'] ?? '';
    $shiftIn = $row['shift_in'] ?? '';
    $token = $row['token'] ?? '';

    if (empty($token)) continue;

    // Step 3: Check outdoor attendance
    $outdoorCheck = mysqli_query($conn, "
        SELECT id FROM outdoor_attendance 
        WHERE user_id='$staffId' AND DATE(shift_date)=CURDATE() 
        LIMIT 1
    ");
    $outdoorExists = mysqli_num_rows($outdoorCheck) > 0;

    // Step 4: Only send notification if no attendance and no outdoor attendance
    $attendanceCheck = mysqli_query($conn, "
        SELECT id FROM attendance 
        WHERE user_id='$staffId' AND DATE(check_in_time)=CURDATE() 
        LIMIT 1
    ");
    $attendanceExists = mysqli_num_rows($attendanceCheck) > 0;

    if (!$attendanceExists && !$outdoorExists) {
        $title = "Missed Check-In";
        $body = "Hi {$staffName}, your shift started at {$shiftIn}. Please check in.";

        $payload = [
            "message" => [
                "token" => $token,
                "notification" => [
                    "title" => $title,
                    "body" => $body
                ],
                "android" => [
                    "notification" => [
                        "channel_id" => "kariigar_notification_channel",
                        "sound" => "notification",
                        "click_action" => "FLUTTER_NOTIFICATION_CLICK",
                        "default_vibrate_timings" => true,
                        "vibrate_timings" => [
                            ["seconds" => 0, "nanos" => 0],
                            ["seconds" => 0, "nanos" => 500000000],
                            ["seconds" => 1, "nanos" => 0]
                        ]
                    ]
                ],
                "data" => [
                    "staff_id" => $staffId,
                    "with_actions" => "true",
                    "actions" => json_encode([
                        ["id" => "busy", "title" => "I'm Busy"],
                        ["id" => "meeting", "title" => "I'm in a Meeting"]
                    ])
                ]
            ]
        ];

        $payloadJson = json_encode($payload, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        $sent = sendFCMv1($payloadJson, $accessToken, $serviceAccount['project_id']);

        $notifications[] = [
            "staff" => $staffName,
            "status" => $sent
        ];
    }
}

echo json_encode([
    "success" => true,
    "sent" => $notifications
]);




// ----------------- FUNCTIONS -----------------

function base64UrlEncode($data) {
    return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
}

function createJWT($serviceAccount) {
    $header = ['alg' => 'RS256', 'typ' => 'JWT'];
    $now = time();
    $claimSet = [
        'iss' => $serviceAccount['client_email'],
        'scope' => 'https://www.googleapis.com/auth/firebase.messaging',
        'aud' => $serviceAccount['token_uri'],
        'iat' => $now,
        'exp' => $now + 3600
    ];

    $base64UrlHeader = base64UrlEncode(json_encode($header));
    $base64UrlClaimSet = base64UrlEncode(json_encode($claimSet));

    $signatureInput = $base64UrlHeader . "." . $base64UrlClaimSet;

    openssl_sign($signatureInput, $signature, $serviceAccount['private_key'], "SHA256");

    $base64UrlSignature = base64UrlEncode($signature);

    return $signatureInput . "." . $base64UrlSignature;
}

function getAccessToken($jwt) {
    $tokenUri = 'https://oauth2.googleapis.com/token';
    $postFields = http_build_query([
        'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
        'assertion' => $jwt
    ]);

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $tokenUri);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/x-www-form-urlencoded'
    ]);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $postFields);

    $response = curl_exec($ch);
    curl_close($ch);

    $data = json_decode($response, true);
    return $data['access_token'] ?? null;
}

function sendFCMv1($payloadJson, $accessToken, $projectId) {
    $url = "https://fcm.googleapis.com/v1/projects/{$projectId}/messages:send";

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        "Authorization: Bearer {$accessToken}",
        "Content-Type: application/json"
    ]);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $payloadJson); // send JSON string directly

    $response = curl_exec($ch);
    if ($response === FALSE) {
        $error = curl_error($ch);
        curl_close($ch);
        return $error;
    }
    curl_close($ch);
    return $response;
}

?>
