<?php
// visitor_entry.php
include("csism_db.php");
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(["success" => false, "message" => "Only POST allowed"]);
    exit();
}

$name        = $_POST['name'] ?? '';
$phone       = $_POST['phone'] ?? '';
$purpose     = $_POST['purpose'] ?? '';
$description = $_POST['description'] ?? '';
$latitude    = $_POST['latitude'] ?? '';
$longitude   = $_POST['longitude'] ?? '';
$timestamp   = $_POST['timestamp'] ?? date("Y-m-d H:i:s");
$userId      = $_POST['user_id'] ?? ''; 

// Validate required fields
if (empty($name) || empty($phone) || empty($purpose) || empty($latitude) || empty($longitude)) {
    echo json_encode(["success" => false, "message" => "Missing required fields"]);
    exit();
}

// Handle photo
$photoUrl = "";
if (isset($_FILES['photo']) && $_FILES['photo']['error'] === UPLOAD_ERR_OK) {
    $uploadsDir = __DIR__ . "/../uploads/visitor_photos/";
    $setDir     = "uploads/visitor_photos/";
        
    if (!is_dir($uploadsDir)) {
        mkdir($uploadsDir, 0777, true);
    }
    
    $fileTmp  = $_FILES['photo']['tmp_name'];
    $fileExt  = strtolower(pathinfo($_FILES['photo']['name'], PATHINFO_EXTENSION));
    $fileName = uniqid("visitor_") . "." . $fileExt;
    $target   = $uploadsDir . $fileName;
    
    // Compress / save
    if (in_array($fileExt, ['jpg', 'jpeg'])) {
        $image = imagecreatefromjpeg($fileTmp);
        $image = fixImageOrientation($image, $fileTmp);
        imagejpeg($image, $target, 70); // 70% quality
        imagedestroy($image);
        $photoUrl = $setDir . $fileName;
    } elseif ($fileExt === 'png') {
        $image = imagecreatefrompng($fileTmp);
        $image = fixImageOrientation($image, $fileTmp);
        imagepng($image, $target, 6);
        imagedestroy($image);
        $photoUrl = $setDir . $fileName;
    } else {
        if (move_uploaded_file($fileTmp, $target)) {
            $photoUrl = $setDir . $fileName;
        }
    }
} else {
    echo json_encode(["success" => false, "message" => "Photo is required"]);
    exit();
}

// Insert into DB
$stmt = $conn->prepare("INSERT INTO visitor_entries 
    (staff_id, name, phone, purpose, description, photo_url, latitude, longitude, entry_timestamp, created_at) 
    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())");

$stmt->bind_param(
    "isssssdss",  // s=string, d=double, i=int
    $userId,
    $name,
    $phone,
    $purpose,
    $description,
    $photoUrl,
    $latitude,
    $longitude,
    $timestamp
);

if ($stmt->execute()) {
    echo json_encode([
        "success" => true,
        "message" => "Visitor entry submitted successfully",
        "data" => [
            "id" => $conn->insert_id,
            "name" => $name,
            "phone" => $phone,
            "purpose" => $purpose,
            "description" => $description,
            "photo_url" => $photoUrl,
            "latitude" => $latitude,
            "longitude" => $longitude,
            "entry_timestamp" => $timestamp
        ]
    ]);
} else {
    echo json_encode(["success" => false, "message" => "Database insert failed: " . $stmt->error]);
}

$stmt->close();
$conn->close();


// -------------------- Helper --------------------
function fixImageOrientation($image, $fileTmp) {
    if (!function_exists('exif_read_data')) return $image;
    $exif = @exif_read_data($fileTmp);
    if (!$exif || !isset($exif['Orientation'])) return $image;
    switch ($exif['Orientation']) {
        case 3: $image = imagerotate($image, 180, 0); break;
        case 6: $image = imagerotate($image, -90, 0); break;
        case 8: $image = imagerotate($image, 90, 0); break;
    }
    return $image;
}
?>
